{*********************************************************************
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Autor: Brovin Y.D.
 * E-mail: y.brovin@gmail.com
 *
 ********************************************************************}

unit FGX.ActionSheet.Android;

interface

uses
  FGX.ActionSheet,
  Androidapi.JNIBridge, Androidapi.JNI.GraphicsContentViewText, Androidapi.JNI.App,
  FGX.ActionSheet.Types;

type

  { TAndroidActionSheetService }

  TActionSheetListener = class;

  TAndroidActionSheetService = class (TInterfacedObject, IFGXActionSheetService)
  private
    FListener: TActionSheetListener;
    FActions: TfgActionsCollections;
  protected
    procedure DoButtonClicked(const ButtonIndex: Integer); virtual;
  public
    constructor Create;
    destructor Destroy; override;

    { IFGXActionSheetService }
    procedure Show(const ATitle: string; Actions: TfgActionsCollections; const UseUIGuidline: Boolean = True);
  end;

  TNotifyButtonClicked = procedure (const ButtonIndex: Integer) of object;

  TActionSheetListener = class (TJavaLocal, JDialogInterface_OnClickListener)
  private
    FOnButtonClicked: TNotifyButtonClicked;
  public
    constructor Create(const AOnButtonClicked: TNotifyButtonClicked);
    { JPopupMenu_OnMenuItemClickListener }
     procedure onClick(dialog: JDialogInterface; which: Integer); cdecl;
  end;

procedure RegisterService;

var
  ActionSheetService: TAndroidActionSheetService;

implementation

uses
  System.Classes, System.Math, System.SysUtils,
  Androidapi.JNI.JavaTypes,
  FGX.Helpers.Android,
  FMX.Platform, FMX.Platform.Android, FMX.Helpers.Android, FMX.Types, FMX.Controls;

procedure RegisterService;
begin
  if TOSVersion.Check(4, 0) then
  begin
    ActionSheetService := TAndroidActionSheetService.Create;
    TPlatformServices.Current.AddPlatformService(IFGXActionSheetService, ActionSheetService);
  end;
end;

{ TAndroidActionSheetService }

constructor TAndroidActionSheetService.Create;
begin
  FListener := TActionSheetListener.Create(DoButtonClicked);
end;

destructor TAndroidActionSheetService.Destroy;
begin
  FListener.Free;
  inherited Destroy;
end;

procedure TAndroidActionSheetService.DoButtonClicked(const ButtonIndex: Integer);
var
  Action: TfgActionCollectionItem;
begin
  Assert(FActions <> nil, 'List of all actions (TActionCollection) already was destroyed');
  Assert(InRange(ButtonIndex, 0, FActions.Count - 1), 'Android returns wrong index of actions. Out of range.');

  if InRange(ButtonIndex, 0, FActions.Count - 1) then
  begin
    Action := (FActions.Items[ButtonIndex] as TfgActionCollectionItem);
    if Assigned(Action.OnClick) then
      Action.OnClick(Action);
  end;
end;

procedure TAndroidActionSheetService.Show(const ATitle: string; Actions: TfgActionsCollections; const UseUIGuidline: Boolean = True);
var
  I: Integer;
  Action: TfgActionCollectionItem;
  DialogBuilder: JAlertDialog_Builder;
  Items: TJavaObjectArray<JCharSequence>;
  Dialog: JAlertDialog;
begin
  Assert(Actions <> nil);

  FActions := Actions;
  { Forming  Action List }
  Items := TJavaObjectArray<JCharSequence>.Create(Actions.Count);
  for I := 0 to Actions.Count - 1 do
  begin
    Action := Actions.GetAction(I);
    if Action.Visible then
      Items.SetRawItem(I, (StrToJCharSequence(Action.Caption) as ILocalObject).GetObjectID);
  end;

  { Create Alert Dialog }
  DialogBuilder := TJAlertDialog_Builder.JavaClass.init(SharedActivityContext, GetNativeTheme);
  if not ATitle.IsEmpty then
    DialogBuilder.setTitle(StrToJCharSequence(ATitle));
  DialogBuilder.setItems(Items, FListener);
  DialogBuilder.setCancelable(True);

  CallInUIThread(procedure begin
    Dialog := DialogBuilder.Create;
    Dialog.Show;
  end);
end;

{ TActionSheetListener }

constructor TActionSheetListener.Create(const AOnButtonClicked: TNotifyButtonClicked);
begin
  inherited Create;
  FOnButtonClicked := AOnButtonClicked;
end;

procedure TActionSheetListener.onClick(dialog: JDialogInterface; which: Integer);
begin
  if Assigned(FOnButtonClicked) then
    TThread.Synchronize(nil, procedure begin
      FOnButtonClicked(which);
    end);
end;

end.
