{*********************************************************************
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Autor: Brovin Y.D.
 * E-mail: y.brovin@gmail.com
 *
 ********************************************************************}

unit FGX.ActionSheet.Types;

interface

uses
  System.Classes;

type

  { TfgActionsCollections }

  TfgActionCollectionItem = class;

  TfgActionStyle = (Normal, Cancel, Destructive);

  TfgActionsCollections = class (TCollection)
  protected
    FOwner: TPersistent;
    function GetOwner: TPersistent; override;
    function IndexOfAction(const Style: TfgActionStyle): Integer; virtual;
  public
    constructor Create(const AOwner: TPersistent);
    /// <summary>
    ///   Return Index of first "destructive" action (action with style as acDestructive)
    ///   If action is not found, return -1
    /// </summary>
    function IndexOfDestructiveButton: Integer;
    function IndexOfCancelButton: Integer;
    function GetAction(const Index: Integer): TfgActionCollectionItem;
  end;

  { TfgActionCollectionItem }

  TfgActionCollectionItem = class (TCollectionItem)
  protected
    FCaption: string;
    FStyle: TfgActionStyle;
    FVisible: Boolean;
    FOnClick: TNotifyEvent;
    procedure AssignTo(Dest: TPersistent); override;
    function Collection: TfgActionsCollections; virtual;
    function GetDisplayName: string; override;
  public
    constructor Create(Collection: TCollection); override;
  published
    property Caption: string read FCaption write FCaption;
    property Style: TfgActionStyle read FStyle write FStyle default TfgActionStyle.Normal;
    property Visible: Boolean read FVisible write FVisible default True;
    property OnClick: TNotifyEvent read FOnClick write FOnClick;
  end;

implementation

uses
  System.SysUtils, System.Math;

{ TfgActionsCollections }

constructor TfgActionsCollections.Create(const AOwner: TPersistent);
begin
  inherited Create(TfgActionCollectionItem);
  FOwner := AOwner;
end;

function TfgActionsCollections.GetAction(const Index: Integer): TfgActionCollectionItem;
begin
  Assert(InRange(Index, 0, Count - 1));
  Result := Items[Index] as TfgActionCollectionItem;
end;

function TfgActionsCollections.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

function TfgActionsCollections.IndexOfAction(const Style: TfgActionStyle): Integer;
var
  Index: Integer;
  Found: Boolean;
begin
  Index := 0;
  Found := False;
  while (Index < Count) and not Found do
    if GetAction(Index).Style = Style then
      Found := True
    else
      Inc(Index);
  if Found then
    Result := Index
  else
    Result := -1;
end;

function TfgActionsCollections.IndexOfCancelButton: Integer;
begin
  Result := IndexOfAction(TfgActionStyle.Cancel);
end;

function TfgActionsCollections.IndexOfDestructiveButton: Integer;
begin
  Result := IndexOfAction(TfgActionStyle.Destructive);
end;

{ TfgActionCollectionItem }

procedure TfgActionCollectionItem.AssignTo(Dest: TPersistent);
var
  DestAction: TfgActionCollectionItem;
begin
  if Dest is TfgActionCollectionItem then
  begin
    DestAction := Dest as TfgActionCollectionItem;
    DestAction.Caption := Caption;
    DestAction.Visible := Visible;
    DestAction.Style := Style;
    DestAction.OnClick := OnClick;
  end
  else
    inherited AssignTo(Dest);
end;

function TfgActionCollectionItem.Collection: TfgActionsCollections;
begin
  Assert(Collection <> nil);
  Result := Collection as TfgActionsCollections;
end;

constructor TfgActionCollectionItem.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FStyle := TfgActionStyle.Normal;
  FVisible := True;
end;

function TfgActionCollectionItem.GetDisplayName: string;
begin
  if Caption.IsEmpty then
    Result := inherited GetDisplayName
  else
    Result := Caption;
end;

end.
