{*********************************************************************
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Autor: Brovin Y.D.
 * E-mail: y.brovin@gmail.com
 *
 ********************************************************************}

 unit FGX.ProgressDialog.Android;

interface

uses
  FGX.ProgressDialog, AndroidApi.ProgressDialog, FGX.ProgressDialog.Types;

type

  { TAndroidProgressDialogService }

  TAndroidProgressDialogService = class (TInterfacedObject, IFGXProgressDialogService)
  public
    { IFGXProgressDialogService }
    function CreateNativeProgressDialog: TfgNativeProgressDialog;
    function CreateNativeActivityDialog: TfgNativeActivityDialog;
  end;

  TAndroidNativeActivityDialog = class (TfgNativeActivityDialog)
  private
    FProgressDialog: JProgressDialog;
  protected
    procedure SetTitle(const Value: string); override;
    procedure SetMessage(const Value: string); override;
  public
    procedure Show; override;
    procedure Hide; override;
  end;

  TAndroidNativeProgressDialog = class (TfgNativeProgressDialog)
  private
    FProgressDialog: JProgressDialog;
  protected
    function IsDialogKindDeterminated(const DialogKind: TfgProgressDialogKind): Boolean;
    { inherited }
    procedure SetTitle(const Value: string); override;
    procedure SetMessage(const Value: string); override;
    procedure SetProgress(const Value: Single); override;
    procedure SetKind(const Value: TfgProgressDialogKind); override;
  public
    procedure Show; override;
    procedure Hide; override;
  end;

procedure RegisterService;

var
  ProgressDialogService: TAndroidProgressDialogService;

implementation

uses
  FMX.Platform, FMX.Helpers.Android, System.SysUtils, FGX.Helpers.Android;

procedure RegisterService;
begin
  if TOSVersion.Check(4, 0) then
  begin
    ProgressDialogService := TAndroidProgressDialogService.Create;
    TPlatformServices.Current.AddPlatformService(IFGXProgressDialogService, ProgressDialogService);
  end;
end;

{ TAndroidProgressDialogService }

function TAndroidProgressDialogService.CreateNativeActivityDialog: TfgNativeActivityDialog;
begin
  Result := TAndroidNativeActivityDialog.Create;
end;

function TAndroidProgressDialogService.CreateNativeProgressDialog: TfgNativeProgressDialog;
begin
  Result := TAndroidNativeProgressDialog.Create;
end;

{ TAndroidNativeProgressDialog }

procedure TAndroidNativeActivityDialog.Hide;
begin
  CallInUIThread(procedure begin
    FProgressDialog.dismiss;
  end);
end;

procedure TAndroidNativeActivityDialog.SetMessage(const Value: string);
begin
  inherited SetMessage(Value);
  if FProgressDialog <> nil then
    CallInUIThread(procedure begin
      FProgressDialog.setMessage(StrToJCharSequence(Message));
    end);
end;

procedure TAndroidNativeActivityDialog.SetTitle(const Value: string);
begin
  inherited SetTitle(Value);
  if FProgressDialog <> nil then
    CallInUIThread(procedure begin
      FProgressDialog.setTitle(StrToJCharSequence(Title));
    end);
end;

procedure TAndroidNativeActivityDialog.Show;
begin
  CallInUIThread(procedure begin
    FProgressDialog := TJProgressDialog.JavaClass.init(SharedActivityContext, GetNativeTheme);
    FProgressDialog.setTitle(StrToJCharSequence(Title));
    FProgressDialog.setMessage(StrToJCharSequence(Message));
    FProgressDialog.setProgressStyle(TJProgressDialog.JavaClass.STYLE_SPINNER);
    FProgressDialog.setCanceledOnTouchOutside(False);

    FProgressDialog.show;
  end);
end;

{ TAndroidNativeActivityDialog }

procedure TAndroidNativeProgressDialog.Hide;
begin
  if FProgressDialog <> nil then
    CallInUIThread(procedure begin
      FProgressDialog.dismiss;
    end);
end;

function TAndroidNativeProgressDialog.IsDialogKindDeterminated(const DialogKind: TfgProgressDialogKind): Boolean;
begin
  Result := DialogKind = TfgProgressDialogKind.Undeterminated;
end;

procedure TAndroidNativeProgressDialog.SetKind(const Value: TfgProgressDialogKind);
begin
  inherited SetKind(Value);
  if FProgressDialog <> nil then
    CallInUIThread(procedure begin
      FProgressDialog.setIndeterminate(IsDialogKindDeterminated(Kind));
    end);
end;

procedure TAndroidNativeProgressDialog.SetMessage(const Value: string);
begin
  inherited SetMessage(Value);
  if FProgressDialog <> nil then
    CallInUIThread(procedure begin
      FProgressDialog.setMessage(StrToJCharSequence(Message));
    end);
end;

procedure TAndroidNativeProgressDialog.SetProgress(const Value: Single);
begin
  inherited SetProgress(Value);
  if FProgressDialog <> nil then
    CallInUIThread(procedure begin
      FProgressDialog.setProgress(Round(Value));
    end);
end;

procedure TAndroidNativeProgressDialog.SetTitle(const Value: string);
begin
  inherited SetTitle(Value);
  if FProgressDialog <> nil then
    CallInUIThread(procedure begin
      FProgressDialog.setTitle(StrToJCharSequence(Title));
    end);
end;

procedure TAndroidNativeProgressDialog.Show;
begin
  CallInUIThread(procedure begin
    FProgressDialog := TJProgressDialog.JavaClass.init(SharedActivityContext, GetNativeTheme);
    FProgressDialog.setTitle(StrToJCharSequence(Title));
    FProgressDialog.setMessage(StrToJCharSequence(Message));
    FProgressDialog.setMax(100);
    FProgressDialog.setProgress(Round(Progress));
    FProgressDialog.setProgressStyle(TJProgressDialog.JavaClass.STYLE_HORIZONTAL);
    FProgressDialog.setIndeterminate(IsDialogKindDeterminated(Kind));
    FProgressDialog.setCanceledOnTouchOutside(False);

    FProgressDialog.show;
  end);
end;

end.
