unit CacheLayout;

interface

uses
  System.SysUtils, System.Classes, System.Types, FMX.Layouts, FMX.Graphics, FMX.Objects, FMX.Types, FMX.Platform,
  FMX.Controls, FMX.StdCtrls, System.UITypes, FMX.Forms, FMX.Effects, FMX.Filter.Effects{$IFNDEF VER270}, FMX.PixelFormats{$ENDIF};

type
  TImageCacheType = (ctNone, ctCacheAsBitmap, ctExportAsBitmap);

  TImageCacheUpdate = (cuMouse, cuTouch, cuTouchDelay, cuManual);

  TImageCacheEffect = (cqNormal, cqNoAlphaEdge, cqSharpen);

  [ComponentPlatformsAttribute(pidWin32 or pidWin64 or pidOSX32 or pidiOSSimulator or pidiOSDevice or pidAndroid)]
  TImageCacheLayout = class(TLayout)
  private
    fCached: Boolean;
    fMakingScreenshot: Boolean;
    fControlClicked: Boolean;                  // Mouse flag to indicate the button state when handling mouse events
    fCacheImage: TImage;
    fCacheType: TImageCacheType;
    fCacheUpdate: TImageCacheUpdate;
    fCacheEffect: TImageCacheEffect;
    fScreenScale: Single;
    fTimer: TTimer;
    fCacheDelay: Integer;
    fEffect: TSharpenEffect;
    procedure ApplySharpenEffect(ABitmap: TBitmap; AAmount: Single);
    function  FindControlAtPoint(aParent: TControl; aPos: TPointF): TControl;
    function  GetScreenScale: Single;
    function  MakeScaleScreenshot: TBitmap;
    procedure ApplyNoAlphaEdge(ABitmap: TBitmap; OpacityThreshold: integer);
    function  GetCacheImage: TBitmap;
    procedure SetCacheImage(const Value: TBitmap);
    procedure SetCacheType(const Value: TImageCacheType);
    procedure SetCacheUpdate(const Value: TImageCacheUpdate);
    procedure ShowChildControls(aShow: Boolean);
    procedure DoTimerDelay;
    procedure TimerInterval(Sender: TObject);
  protected
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Single); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Single); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Single); override;
    procedure DoMouseLeave; override;
    procedure Paint; override;
    procedure Resize; override;
  public
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
    procedure ClearCachedImage;
    property CacheDelay: Integer read fCacheDelay write fCacheDelay default 5000;
  published
    property CacheImage: TBitmap read GetCacheImage write SetCacheImage;
    property CacheType: TImageCacheType read fCacheType write SetCacheType;
    property CacheUpdate: TImageCacheUpdate read fCacheUpdate write SetCacheUpdate;
    property CacheEffect: TImageCacheEffect read fCacheEffect write fCacheEffect;
  end;

implementation

type
  THackControl = type TControl;

// Find a child control that is under the given coordinates (form coordinates)
// The control must be able to accept mouse events, e.g. HitTest=True
function TImageCacheLayout.FindControlAtPoint(aParent: TControl; aPos: TPointF): TControl;
var
  I: Integer;
  Control, ChildControl: TControl;
  S: String;
begin
  Result := nil;

  // Check all the child controls and find the one at the coordinates
  for I := aParent.Controls.Count - 1 downto 0 do
  begin
    Control := aParent.Controls[I];
    S := Control.ClassName;
    if Control.PointInObject(aPos.X, aPos.Y) and (Control <> fCacheImage) then
    begin
      ChildControl := FindControlAtPoint(Control, aPos);
      if Assigned(ChildControl) and ChildControl.HitTest then
        Exit(ChildControl)
      else
        if Control.HitTest then
          Exit(Control);
    end;
  end;
end;

// Get the screen scale factor used to generate the bitmap
function TImageCacheLayout.GetScreenScale: Single;
var
   ScreenService: IFMXScreenService;
begin
   Result := 1;
   if TPlatformServices.Current.SupportsPlatformService (IFMXScreenService, IInterface(ScreenService)) then
   begin
      Result := ScreenService.GetScreenScale;
   end;
end;

function TImageCacheLayout.MakeScaleScreenshot: TBitmap;
begin
  Result := TBitmap.Create(Round(Width*fScreenScale), Round(Height*fScreenScale));
  Result.Clear(0);
  if Result.Canvas.BeginScene then
    try
      PaintTo(Result.Canvas, RectF(0,0,Result.Width,Result.Height), Self.Parent);
      case fCacheEffect of
       cqNormal: begin end;
       cqNoAlphaEdge: ApplyNoAlphaEdge(Result,63);
       cqSharpen: ApplySharpenEffect(Result,0.01);
      end;
    finally
      Result.Canvas.EndScene;
    end;
end;

procedure TImageCacheLayout.ApplyNoAlphaEdge(ABitmap: TBitmap; OpacityThreshold: integer);
var
  bitdata1: TBitmapData;
  I: integer;
  J: integer;
  C: TAlphaColor;
begin
  if (ABitmap.Map(TMapAccess.maReadWrite, bitdata1)) then
    try
      for I := 0 to ABitmap.Width - 1 do
        for J := 0 to ABitmap.Height - 1 do
        begin
          begin
              {$IFDEF VER270}
              C := PixelToAlphaColor(@PAlphaColorArray(bitdata1.Data)
              [J * (bitdata1.Pitch div PixelFormatBytes[ABitmap.PixelFormat])
               + 1 * I], ABitmap.PixelFormat);
              {$ELSE}
              C := PixelToAlphaColor(@PAlphaColorArray(bitdata1.Data)
              [J * (bitdata1.Pitch div GetPixelFormatBytes(ABitmap.PixelFormat))
               + 1 * I], ABitmap.PixelFormat);
              {$ENDIF}
               if TAlphaColorRec(C).A<OpacityThreshold then
                begin
                  TAlphaColorRec(C).A := 0;

              {$IFDEF VER270}
                  AlphaColorToPixel(C, @PAlphaColorArray(bitdata1.Data)
                    [J * (bitdata1.Pitch div PixelFormatBytes[ABitmap.PixelFormat])
                     + 1 * I], ABitmap.PixelFormat);
              {$ELSE}
                  AlphaColorToPixel(C, @PAlphaColorArray(bitdata1.Data)
                    [J * (bitdata1.Pitch div GetPixelFormatBytes(ABitmap.PixelFormat))
                     + 1 * I], ABitmap.PixelFormat);
              {$ENDIF}
                end;
          end;
        end;
    finally
      ABitmap.Unmap(bitdata1);
    end;
end;

procedure  TImageCacheLayout.ApplySharpenEffect(ABitmap: TBitmap; AAmount: Single);
begin
  fEffect.Amount := AAmount;
  fEffect.ProcessEffect(ABitmap.Canvas, ABitmap, 1);
end;

function TImageCacheLayout.GetCacheImage: TBitmap;
begin
  Result := fCacheImage.Bitmap;
end;

procedure TImageCacheLayout.SetCacheImage(const Value: TBitmap);
begin
  fCacheImage.Bitmap.Assign(Value);
end;

procedure TImageCacheLayout.SetCacheType(const Value: TImageCacheType);
begin
  fCacheType := Value;

  // Don't do any processing or clear the image on the initial loading or at design-time the component
  if (csLoading in ComponentState) or (csDesigning in ComponentState) then
    Exit;

  case Value of
    ctNone:
      begin
        // Reset the image when changing the CacheType
        ClearCachedImage;
        // Show the original child controls, if any
        ShowChildControls(true);
      end;
    // Call Paint to process the image
    ctCacheAsBitmap:
      begin
        Paint;
      end;
    // Call Paint to process the image
    ctExportAsBitmap:
      begin
        Paint;
      end;
  end;
end;

procedure TImageCacheLayout.SetCacheUpdate(const Value: TImageCacheUpdate);
begin
  fCacheUpdate := Value;

  // Don't do any processing or clear the image on the initial loading or at design-time the component
  if (csLoading in ComponentState) or (csDesigning in ComponentState) then
    Exit;

  case Value of
    cuManual:
      begin
       HitTest := False;
      end;
    else
     begin
       HitTest := True;
     end;
  end;
end;

// Switch between showing the child controls or the TImage
procedure TImageCacheLayout.ShowChildControls(aShow: Boolean);
var
  I: Integer;
begin
  // Show(or hide) the original child controls
  for I := 0 to Controls.Count - 1 do
    if Controls[I] <> fCacheImage then
      Controls[I].Visible := aShow;
  // Hide(or show) the CacheImage
  fCacheImage.Visible := not aShow;
end;

// To update the underlying controls capture the MouseDown event, find the control under the cursor
// and forward the event to it
// Unhide the controls temporarily to allow them to refresh and then recache the image
procedure TImageCacheLayout.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Single);
var
  Control: TControl;
begin
  inherited;

  if (CacheUpdate = cuManual) or (CacheType <> ctCacheAsBitmap) or (csDesigning in ComponentState) then
    Exit;

  // Locate the control below the cursor
  Control := FindControlAtPoint(Self, LocalToAbsolute(PointF(X, Y)));
  if Assigned(Control) and (TControl(Control) <> Self) then
  begin
    fControlClicked := true;
    ShowChildControls(true);
    THackControl(Control).MouseDown(Button, Shift, X, Y);
  end;

  // Finish the update immediately if the cuTouch option is selected
  if CacheUpdate = cuTouch then
    DoMouseLeave;

  if CacheUpdate = cuTouchDelay then
    DoTimerDelay;
end;

// Forward the mouse move events to underlying controls, if they are shown
procedure TImageCacheLayout.MouseMove(Shift: TShiftState; X, Y: Single);
var
  Control: TControl;
begin
  inherited;

  if (CacheUpdate = cuManual) or (CacheType <> ctCacheAsBitmap) or (csDesigning in ComponentState) then
    Exit;
  if not fControlClicked then
    Exit;

  // Locate the control below the cursor
  Control := FindControlAtPoint(Self, LocalToAbsolute(PointF(X, Y)));
  if Assigned(Control) and (TControl(Control) <> Self) then
    THackControl(Control).MouseMove(Shift, X, Y);
end;

// Forward mouse up events to the underlying controls, if they are shown
procedure TImageCacheLayout.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Single);
var
  Control: TControl;
begin
  inherited;
  if (CacheUpdate = cuManual) or (CacheType <> ctCacheAsBitmap) or (csDesigning in ComponentState) then
    Exit;

  // Locate the control below the cursor
  Control := FindControlAtPoint(Self, LocalToAbsolute(PointF(X, Y)));
  if Assigned(Control) and (TControl(Control) <> Self) then
    THackControl(Control).MouseUp(Button, Shift, X, Y);

  if CacheUpdate = cuTouch then
    DoMouseLeave;
end;

// When the mouse leaves the layout cache the image and hide the controls
// If the CacheUpdate is cuTouch do the updating immediately after the control is clicked
procedure TImageCacheLayout.DoMouseLeave;
var
  Pos: TPointF;
  MouseInside: Boolean;
begin
  inherited;

  if (CacheUpdate = cuManual) or (CacheUpdate = cuTouchDelay) or (CacheType <> ctCacheAsBitmap) or (csDesigning in ComponentState) then
    Exit;

  // Check if the event is called because another control got the focus inside the layout, or the mouse really left it
  // The image is refreshed if the mouse left, or if the CacheUpdate is cuTouch
  Pos := ScreenToLocal(IFMXMouseService(TPlatformServices.Current.GetPlatformService(IFMXMouseService)).GetMousePos);
  MouseInside := PointInObjectLocal(Pos.X, Pos.Y);
  if fControlClicked and (not MouseInside or (CacheUpdate = cuTouch)) then
  begin
    fControlClicked := false;
    ShowChildControls(true);
    ClearCachedImage;
    Paint;
  end;
end;

// Displays the cached image or makes a new one if there is no cache
// Also hides or frees the other child components depending on the cache type
procedure TImageCacheLayout.Paint;
var
  Image: TBitmap;
  I: Integer;
begin
  inherited;

  // If the image is already generated and displayed there is no need to do the processing again
  // Also check if a MakeScreenshot call is in progress (the fMakingScreenshot flag is set)
  // The MakeScreenshot internally calls Paint, which will lead to an endless recursion and a stack overflow if
  // the flag is not set
  if fCached or (CacheType = ctNone) or fMakingScreenshot then
    Exit;

  // The Paint provides only run-time functionality
  if not (csDesigning in ComponentState) then
  begin
    // Create a new image
    if fCacheImage.Bitmap.IsEmpty then
    begin
      try
        // Set the flag that we are taking a screenshot to avoid further processing when MakeScreenshot calls Paint again
        fMakingScreenshot := true;
        Image := Self.MakeScaleScreenshot;
        try
          fCacheImage.Bitmap.Assign(Image);
        finally
          Image.Free;
        end;
      finally
        // Reset the flag so the Paint method can continue normally next time
        fMakingScreenshot := false;
      end;
    end;

    // Hide or remove the other child components, depending on the cache type
    case CacheType of
      ctCacheAsBitmap:
        begin
          ShowChildControls(false);
        end;
      ctExportAsBitmap:
        begin
          // The new moile compiler uses ARC (auto reference counting) and objects are freed autmatically when the refcount reaches 0
          for I := Self.Controls.Count - 1 downto 0 do
            Self.Controls[I].DisposeOf;
        end;
    end;

    // Show the cache image on top of everything else
    fCacheImage.Parent := Self;
    fCacheImage.BringToFront;

    fCached := true;
  end;
end;

// Refresh the image on resize if the type is ctCaCacheAsBitmap. In other types there is nothing to refresh
procedure TImageCacheLayout.Resize;
begin
  inherited;

  if (CacheType <> ctCacheAsBitmap) or (csDesigning in ComponentState) or (not fCached) then
    Exit;

  ShowChildControls(true);
  ClearCachedImage;
end;

procedure TImageCacheLayout.DoTimerDelay;
begin
  if (CacheType = ctNone) or fMakingScreenshot or (csDesigning in ComponentState) then
    Exit;

  fTimer.Interval := fCacheDelay;
  fTimer.Enabled := True;
end;

procedure TImageCacheLayout.TimerInterval(Sender: TObject);
begin
  fTimer.Enabled := False;

  if (CacheUpdate = cuManual) or (CacheType <> ctCacheAsBitmap) or (csDesigning in ComponentState) then
    Exit;

  fControlClicked := false;
  ShowChildControls(true);
  ClearCachedImage;
  Paint;

end;

constructor TImageCacheLayout.Create(aOwner: TComponent);
begin
  inherited;
  if (CacheUpdate <> cuManual) then
    HitTest := True; // To capture clicks and forward them to child controls

  fScreenScale := GetScreenScale;
  fCacheDelay := 5000;

  fTimer := TTimer.Create(nil);
  fTimer.Enabled := False;
  fTimer.OnTimer := TimerInterval;

  fEffect := TSharpenEffect.Create(nil);

  // Create the TImage control that will display the cached image
  fCacheImage := TImage.Create(nil);
  fCacheImage.HitTest := false;
  fCacheImage.Align := TAlignLayout.alContents;
  // iwStretch has a bit of a blur to it.
  // iwOriginal is not painted with the original scale on iOS/Android
  fCacheImage.WrapMode := TImageWrapMode.iwStretch;

end;

destructor TImageCacheLayout.Destroy;
begin
  fTimer.Free;
  fEffect.Free;
  fCacheImage.Free;
  inherited;
end;

// Delete the contents of the cached image and reset the flag
procedure TImageCacheLayout.ClearCachedImage;
begin
  fCacheImage.Bitmap.Width := 0;
  fCacheImage.Bitmap.Height := 0;
  fCached := false;
end;

end.
