unit CacheLayoutEditors;

interface

uses
  DesignEditors, TypInfo, DesignIntf,
  Vcl.Graphics, VCLEditors,
  System.Classes, System.SysUtils, System.Types, System.UITypes,
  FMX.Graphics, FMX.Layouts, FMX.Dialogs, FMX.Objects;

type
  // The enum property type containing the None/CacheAB/Export cache types
  TCacheTypeProperty = class(TEnumProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  // Property type containing the generated layout image
  // The interfaces are needed to draw the film strip icon next to the CacheImage property in the object inspector
  TCacheImageProperty = class(TPropertyEditor, ICustomPropertyDrawing, ICustomPropertyDrawing80)
  public
    function GetAttributes: TPropertyAttributes; override;
    function GetValue: string; override;
    procedure Edit; override;
    {ICustomPropertyDrawing}
    procedure PropDrawName(ACanvas: Vcl.Graphics.TCanvas; const ARect: TRect; ASelected: Boolean);
    procedure PropDrawValue(ACanvas: Vcl.Graphics.TCanvas; const ARect: TRect; ASelected: Boolean);
    { ICustomPropertyDrawing80 }
    function PropDrawNameRect(const ARect: TRect): TRect;
    function PropDrawValueRect(const ARect: TRect): TRect;
  end;

  // The enum property type containing the update type Mouse/Touch/Manual
  TCacheUpdateProperty = class(TEnumProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  // Context menu editor for the component. Currently provides only "Update image" and "Convert" options
  TImageCacheLayoutEditor = class(TComponentEditor)
  private
    procedure ConvertLayout;
  public
    function GetVerbCount: Integer; override;
    function GetVerb(Index: Integer): String; override;
    procedure ExecuteVerb(Index: Integer); override;
  end;

procedure Register;

implementation

uses
  CacheLayoutForm, CacheLayout;

const
  // Used by the TCacheImageProperty for drawing the small film strip icon next to the CacheImage property in the object inspector
  FilmStripMargin = 2;
  FilmStripWidth  = 12;
  FilmStripHeight = 13;
  FilmStripIsAnimated = false;  // Draws two different types of film strip icons (a lighter or darker one)


{ TCacheTypeProperty }

function TCacheTypeProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList]; // Enum drop down list
end;

procedure TCacheTypeProperty.GetValues(Proc: TGetStrProc);
begin
  Proc('ctNone');
  Proc('ctCacheAsBitmap');
  Proc('ctExportAsBitmap');
end;


{ TCacheImageProperty }

function TCacheImageProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paReadOnly];  // The "..." button to open the Image generation form
end;

// Set the value shown in the Obj Inspector
function TCacheImageProperty.GetValue: string;
var
  Layout: TImageCacheLayout;
begin
  Layout := TImageCacheLayout(GetComponent(0));
  if Layout.CacheImage.IsEmpty then
    Result := '(Bitmap Empty)'
  else
    Result := Format('(Bitmap %d x %d)', [Layout.CacheImage.Width, Layout.CacheImage.Height]);
end;

// Show the image generator form and notify the IDE that the project is modified
procedure TCacheImageProperty.Edit;
begin
  if TfrmCacheImageGenerator.ShowGenerator(TImageCacheLayout(GetComponent(0))) then
    Modified;
end;

procedure TCacheImageProperty.PropDrawName(ACanvas: Vcl.Graphics.TCanvas; const ARect: TRect; ASelected: Boolean);
begin
  DefaultPropertyDrawName(Self, ACanvas, ARect);
end;

// Draw the film-strip icon next to the property in the Object inspector
procedure TCacheImageProperty.PropDrawValue(ACanvas: Vcl.Graphics.TCanvas; const ARect: TRect; ASelected: Boolean);
var
  I, Right, Left, Top: Integer;
  OldPenColor, OldBrushColor: TColor;
  BorderColor, CellColor: TColor;
begin
  Left := ARect.Left + FilmStripMargin;
  Right := Left + FilmStripWidth;
  Top := ARect.Top +  Round((ARect.Bottom - ARect.Top - FilmStripHeight)/2);
  with ACanvas do
  begin
    // save off things
    OldPenColor := Pen.Color;
    OldBrushColor := Brush.Color;

    // frame things
    if FilmStripIsAnimated then
    begin
      BorderColor := TColors.Black;
      CellColor := TColors.LtGray;
    end
    else
    begin
      BorderColor := TColors.LtGray;
      CellColor := TColors.White;
    end;

    Pen.Color := BorderColor;
    Rectangle(Left, Top, Right, Top + FilmStripHeight);
    for I := 0 to 2 do
    begin
      Rectangle(Left, Top + 2 + (4 * I), Right, Top + 5 + (4 * I));
    end;
    Rectangle(Left + 2, Top, Right - 2, Top + FilmStripHeight);

    Brush.Color := CellColor;
    Pen.Color := CellColor;
    Rectangle(Left + 3, Top, Right - 3, Top + FilmStripHeight);

    Pen.Color := BorderColor;
    Rectangle(Left + 2, Top + 3, Right - 2, Top + FilmStripHeight - 3);

    // restore the things we twiddled with
    Brush.Color := OldBrushColor;
    Pen.Color := OldPenColor;
  end;
end;

function TCacheImageProperty.PropDrawNameRect(const ARect: TRect): TRect;
begin
  Result:= ARect;
end;

// The rectangle used for the custom film strip icon
function TCacheImageProperty.PropDrawValueRect(const ARect: TRect): TRect;
begin
  Result := Rect(ARect.Left, ARect.Top, FilmStripMargin * 2 + FilmStripWidth + ARect.Left, ARect.Bottom);
end;


{ TCacheUpdateProperty }

function TCacheUpdateProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList]; // Enum drop down list
end;

procedure TCacheUpdateProperty.GetValues(Proc: TGetStrProc);
begin
  Proc('cuMouse');
  Proc('cuTouch');
  Proc('cuManual');
end;


{ TImageCacheLayoutEditor }

function TImageCacheLayoutEditor.GetVerbCount: Integer;
begin
  // Only two menu items
  Result := 2;
end;

// Build the component context menu
function TImageCacheLayoutEditor.GetVerb(Index: Integer): String;
begin
  case Index of
    0: Result := 'Update Cache Image';
    1: Result := 'Convert To TImage Bitmap';
  else
    raise Exception.Create('The TImageCacheLayout does not support this menu verb');
  end;
end;

procedure TImageCacheLayoutEditor.ExecuteVerb(Index: Integer);
begin
  case Index of
  0:  // The "Update image" menu item
    begin
      // Notify the IDE that the project is modified
      if TfrmCacheImageGenerator.ShowGenerator(TImageCacheLayout(Component)) then
        Designer.Modified;
    end;
  1:
    begin
      if MessageDlg('You are about to delete the layout and convert it to a TImage. Do you want to continue?', TMsgDlgType.mtWarning,
           [TMsgDlgBtn.mbYes, TMsgDlgBtn.mbNo], 0) = mrYes then
        if TfrmCacheImageGenerator.ShowGenerator(TImageCacheLayout(Component)) then
        begin
          // Convert and notify the IDE that the project is modified
          ConvertLayout;
          Designer.Modified;
        end;
    end;
  else
    raise Exception.Create('The TImageCacheLayout does not support this menu verb');
  end;
end;

// Convert the layout and replace it with a TImage
procedure TImageCacheLayoutEditor.ConvertLayout;
var
  OldControl: TImageCacheLayout;
  NewControl: TImage;
  OldName: String;
begin
  OldControl := TImageCacheLayout(Component);
  OldName := OldControl.Name;

  // Create the new control that will hold the image
  NewControl := TImage(Designer.CreateComponent(TImage, OldControl.Parent, 0, 0, 100, 100));

  // Copy the image
  NewControl.Bitmap.Assign(OldControl.CacheImage);

  // Copy some properties from the old control
  NewControl.Align          := OldControl.Align;
  NewControl.Anchors        := OldControl.Anchors;
  NewControl.Cursor         := OldControl.Cursor;
  NewControl.Margins        := OldControl.Margins;
  NewControl.Padding        := OldControl.Padding;
  NewControl.Position       := OldControl.Position;
  NewControl.RotationAngle  := OldControl.RotationAngle;
  NewControl.RotationCenter.Assign(OldControl.RotationCenter);  // This TPosition property does not have a setter method so := copies the object reference and causes AV
  NewControl.Scale          := OldControl.Scale;
  NewControl.Width          := OldControl.Width;
  NewControl.Height         := OldControl.Height;
  NewControl.Visible        := OldControl.Visible;

  // Remove the old control from the form
  Designer.SelectComponent(OldControl);
  Designer.DeleteSelection(true);

  // Select the new one and name it as the old one
  Designer.SelectComponent(NewControl);
  NewControl.Name := OldName;
end;


// Register the component and its properties and editors
procedure Register;
begin
  RegisterPropertyEditor(TypeInfo(TBitmap), TImageCacheLayout, 'CacheImage', TCacheImageProperty);
  RegisterPropertyEditor(TypeInfo(TImageCacheType), TImageCacheLayout, 'CacheImage', TCacheTypeProperty);
  RegisterComponentEditor(TImageCacheLayout, TImageCacheLayoutEditor);
  RegisterComponents('Samples', [TImageCacheLayout]);
end;

end.
