unit CacheLayoutForm;

interface

uses
  CacheLayout,
  System.SysUtils, System.Types, System.UITypes, System.Classes, System.Variants, 
  FMX.Types, FMX.Graphics, FMX.Controls, FMX.Forms, FMX.Dialogs, FMX.StdCtrls,
  FMX.Layouts, FMX.Objects;

type
  TfrmCacheImageGenerator = class(TForm)
    pnlHolder: TPanel;
    ImagePreview: TImage;
    btnOK: TButton;
    btnCancel: TButton;
    cbPreview: TCheckBox;
    ImageLayout: TLayout;
    procedure FormPaint(Sender: TObject; Canvas: TCanvas; const ARect: TRectF);
    procedure cbPreviewChange(Sender: TObject);
  private
    procedure ResizeLayout(aWidth, aHeight: Single);
    procedure CopyProperties(const aSource, aDestination: TControl);
    procedure CopyControls(const aSource, aDestination: TControl);
    procedure GeneratePreview;
  public
    class function ShowGenerator(aLayout: TImageCacheLayout): Boolean;
  end;

implementation

{$R *.fmx}

const
  MIN_FORM_WIDTH  = 300;
  MIN_FORM_HEIGHT = 100;

// Generate the preview in the first OnPaint event when all the controls are available and remove the
// event handler to avoid unnecessary processing when OnPaint is called after that
// This can not be done in OnCreate/Show/Activate because not all controls are rendered at that time
procedure TfrmCacheImageGenerator.FormPaint(Sender: TObject; Canvas: TCanvas; const ARect: TRectF);
begin
  GeneratePreview;
  Self.OnPaint := nil;
end;

// Switch between the screenshot and the copy of the original controls
// This is just for testing and could be removed if not needed in the finished component
procedure TfrmCacheImageGenerator.cbPreviewChange(Sender: TObject);
begin
  ImageLayout.Visible  := not cbPreview.IsChecked;
  ImagePreview.Visible := cbPreview.IsChecked;
end;

// Resize the form so the layout matches the given dimensions. Take into account the margins and borders, if any
procedure TfrmCacheImageGenerator.ResizeLayout(aWidth, aHeight: Single);
begin
  pnlHolder.Width  := aWidth  + ImageLayout.Margins.Left + ImageLayout.Margins.Right;
  pnlHolder.Height := aHeight + ImageLayout.Margins.Top  + ImageLayout.Margins.Bottom;
  Self.ClientWidth  := Round(pnlHolder.Width  + pnlHolder.Margins.Left + pnlHolder.Margins.Right);
  Self.ClientHeight := Round(pnlHolder.Height + pnlHolder.Margins.Top  + pnlHolder.Margins.Bottom);

  // Set some minimum sizes, so the buttons are not cropped
  if Self.ClientWidth < MIN_FORM_WIDTH then
    Self.ClientWidth := MIN_FORM_WIDTH;
  if Self.ClientHeight < MIN_FORM_HEIGHT then
    Self.ClientHeight := MIN_FORM_HEIGHT;
end;

// Helper method to copy all the poperties from one object to another. Used when duplicating the original layout controls
procedure TfrmCacheImageGenerator.CopyProperties(const aSource, aDestination: TControl);
var
  Stream: TMemoryStream;
begin
  if not Assigned(aSource) or not Assigned(aDestination) then
    raise Exception.Create('Properties can not be copied for NIL objects');

  Stream := TMemoryStream.Create;
  try
    Stream.WriteComponent(aSource);
    Stream.Position := 0;
    Stream.ReadComponent(aDestination);
  finally
    Stream.Free;
  end;
end;

// Helper method to copy all the child controls from one parent to another
procedure TfrmCacheImageGenerator.CopyControls(const aSource, aDestination: TControl);
type
  TControlClass = class of TControl;
var
  I: Integer;
  OrigControl, NewControl: TControl;
begin
  for I := 0 to aSource.ControlsCount - 1 do
  begin
    OrigControl := aSource.Controls[I];
    NewControl := TControlClass(OrigControl.ClassType).Create(aDestination);
    CopyProperties(OrigControl, NewControl);
    NewControl.Parent := aDestination;
  end;
end;

// Generate the image from the layout, show it, and hide the original layout
procedure TfrmCacheImageGenerator.GeneratePreview;
var
  fGeneratedImage: TBitmap;
begin
  fGeneratedImage := ImageLayout.MakeScreenshot;
  try
    ImagePreview.Bitmap.Assign(fGeneratedImage);
  finally
    fGeneratedImage.Free;
  end;
  ImageLayout.Visible := false;
  ImagePreview.Visible := true;
end;

// Show the image generation form, create the image and return True if the user presses the OK button
class function TfrmCacheImageGenerator.ShowGenerator(aLayout: TImageCacheLayout): Boolean;
var
  GenForm: TfrmCacheImageGenerator;
begin
  // Create and show the form and process the generated image
  GenForm := TfrmCacheImageGenerator.Create(nil);
  try
    GenForm.ResizeLayout(aLayout.Width, aLayout.Height);

    GenForm.CopyControls(aLayout, GenForm.ImageLayout);
    Result := GenForm.ShowModal = mrOK;
    if Result then
    begin
      aLayout.CacheImage := GenForm.ImagePreview.Bitmap;
      if aLayout.CacheType = ctNone then
        aLayout.CacheType := ctCacheAsBitmap;
    end;
  finally
    GenForm.Release;
  end;
end;

end.
